use master
GO

set nocount on 

--set to 0 to skip updating the A4Survey_Client_Template database, or to 1 to update the template database
declare @ProcessTemplateDB bit = 1  

declare @ClientPayload01 nvarchar(max) = N'
--66455
IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''[dbo].[A4S_sp_ReplicatePanelDatabase]'') AND type in (N''P'', N''PC''))
BEGIN
   DROP PROCEDURE [dbo].[A4S_sp_ReplicatePanelDatabase]
END
'

declare @ClientPayload02 nvarchar(max) = N'
CREATE PROCEDURE [dbo].[A4S_sp_ReplicatePanelDatabase]
AS
BEGIN
  SET NOCOUNT ON
  DECLARE @surveyDatabase nvarchar(64) = DB_NAME()
  DECLARE @clientId int = dbo.GetClientId(@surveyDatabase) 
  DECLARE @clientIdString nvarchar(8) = CONVERT(nvarchar, @clientId)
  DECLARE @panelDatabase nvarchar(40) = ''A4Panel_Client_'' + @clientIdString
  DECLARE @surveyId int
  DECLARE @surveySchema nvarchar(40)
  DECLARE @synchronizationVersion int 
  DECLARE @sqlcmd nvarchar(max) 

  -- Get last version from DB
  SELECT TOP 1 @synchronizationVersion = [Version] FROM [ChangeTrackingVersions] ORDER BY [Version] DESC
 
  IF @synchronizationVersion < CHANGE_TRACKING_CURRENT_VERSION()
  BEGIN 
   BEGIN TRY
    SET TRANSACTION ISOLATION LEVEL SNAPSHOT;
    BEGIN TRAN

    PRINT ''Updating version: '' + CAST(@synchronizationVersion AS NVARCHAR)   

    IF (SELECT COUNT(*) FROM sys.change_tracking_tables WHERE min_valid_version > @synchronizationVersion AND min_valid_version > begin_version) > 0
    BEGIN
   RAISERROR (N''Synchronization version is too old - Call A4S_sp_ResetPanelDatabase to reset all the data.'', 16, -1)
    END
    ELSE
    BEGIN
  -- Schemas
  DECLARE surveyCursor CURSOR LOCAL FOR SELECT * FROM A4S_fn_GetPanelSurveyIds()
    
  OPEN surveyCursor;
  FETCH NEXT FROM surveyCursor INTO @surveyId;
  WHILE @@FETCH_STATUS = 0 BEGIN
   DECLARE @surveyIdString nvarchar(10) = CONVERT(nvarchar, @surveyId) 
   SET @surveySchema = ''Survey_'' + @clientIdString + ''_'' + @surveyIdString
     
    ----------- Respondents 
    -- Insert   
    PRINT ''Replicating ['' + @surveySchema + ''].[Respondent] - Inserted''
    SET @sqlcmd = N''
    INSERT INTO ['' + @panelDatabase + ''].[dbo].[PanelistRespondents] ([SurveyId], [PanelistId], [RespondentId], [RespondentActive], [RespondentOnline])
    SELECT @surveyId, r.[panelistId], r.[ResRespondent], r.[ResActive], r.[ResOnline]
    FROM ['' + @surveySchema + ''].[Respondent] r 
    INNER JOIN ['' + @panelDatabase +  ''].[dbo].[Panelists] p ON r.[panelistId] = p.[Id]
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Respondent], @synchronizationVersion) AS ct
        ON r.[ResRespondent] = ct.[ResRespondent]
        AND ct.SYS_CHANGE_OPERATION = ''''I''''
    LEFT JOIN ['' + @panelDatabase + ''].[dbo].[PanelistRespondents] pr
        ON pr.[SurveyId] = @surveyId
        AND r.[panelistId] = pr.[PanelistId]
        AND r.[ResRespondent] = pr.[RespondentId]
    WHERE pr.[SurveyId] IS NULL''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    -- Update
    PRINT ''Replicating ['' + @surveySchema + ''].[Respondent] - Updated''
    SET @sqlcmd = N''
    DECLARE @ResActiveColumnId int = COLUMNPROPERTY(OBJECT_ID(''''['' + @surveySchema + ''].[Respondent]''''),''''ResActive'''', ''''ColumnId'''');
    DECLARE @ResOnlineColumnId int = COLUMNPROPERTY(OBJECT_ID(''''['' + @surveySchema + ''].[Respondent]''''),''''ResOnline'''', ''''ColumnId'''');

    UPDATE pr
    SET pr.[RespondentActive] = r.[ResActive], pr.[RespondentOnline] = r.[ResOnline]
    FROM ['' + @panelDatabase + ''].[dbo].[PanelistRespondents] AS pr
    INNER JOIN ['' + @surveySchema + ''].[Respondent] AS r ON pr.[RespondentId] = r.[ResRespondent]
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Respondent], @synchronizationVersion) AS ct
        ON r.[ResRespondent] = ct.[ResRespondent]
        AND ct.SYS_CHANGE_OPERATION = ''''U''''
        AND (
            CHANGE_TRACKING_IS_COLUMN_IN_MASK(@ResActiveColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@ResOnlineColumnId, ct.SYS_CHANGE_COLUMNS) = 1
        )
    WHERE pr.[SurveyId] = @surveyId''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    -- Delete
    PRINT ''Replicating ['' + @surveySchema + ''].[Respondent] - Deleted''
    SET @sqlcmd = N''
    DELETE pr
    FROM ['' + @panelDatabase + ''].[dbo].[PanelistRespondents] AS pr
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Respondent], @synchronizationVersion) AS ct
        ON pr.[RespondentId] = ct.[ResRespondent]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''
    WHERE pr.[SurveyId] = @surveyId''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    ----------- Responses
    -- Insert   
    PRINT ''Replicating ['' + @surveySchema + ''].[Response] - Inserted''
    SET @sqlcmd = N''
    INSERT INTO ['' + @panelDatabase + ''].[dbo].[PanelistSurveyResponses] ([SurveyId], [RespondentId], [PanelistId], [VariableId], [VariableName], [Matrix], [Mention], [Content])
    SELECT @surveyId, r.[RpsRespondent], pr.[PanelistId], r.[RpsVariableId], r.[RpsQuestion], r.[RpsMatrix], r.[RpsMention], r.[RpsContent]
    FROM ['' + @surveySchema + ''].[Response] AS r
    INNER JOIN ['' + @panelDatabase +  ''].[dbo].[PanelistRespondents] pr
        ON pr.[SurveyId] = @surveyId
        AND r.[RpsRespondent] = pr.[RespondentId]
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Response], @synchronizationVersion) AS ct
        ON r.[RpsRespondent] = ct.[RpsRespondent]
        AND r.[RpsQuestion] = ct.[RpsQuestion]
        AND r.[RpsMatrix] = ct.[RpsMatrix]
        AND r.[RpsMention] = ct.[RpsMention]
        AND ct.SYS_CHANGE_OPERATION = ''''I''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    -- Update
    PRINT ''Replicating ['' + @surveySchema + ''].[Response] - Updated''
    SET @sqlcmd = N''
    DECLARE @RpsContentColumnId int = COLUMNPROPERTY(OBJECT_ID(''''['' + @surveySchema + ''].[Response]''''),''''RpsContent'''', ''''ColumnId'''');

    UPDATE psr
    SET psr.[Content] = r.[RpsContent]
    FROM ['' + @panelDatabase + ''].[dbo].[PanelistSurveyResponses] AS psr
    INNER JOIN ['' + @surveySchema + ''].[Response] AS r 
        ON psr.[RespondentId] = r.[RpsRespondent]
        AND psr.[VariableName] = r.[RpsQuestion]
        AND psr.[Matrix] = r.[RpsMatrix]
        AND psr.[Mention] = r.[RpsMention]
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Response], @synchronizationVersion) AS ct
        ON r.[RpsRespondent] = ct.[RpsRespondent]
        AND r.[RpsQuestion] = ct.[RpsQuestion]
        AND r.[RpsMatrix] = ct.[RpsMatrix]
        AND r.[RpsMention] = ct.[RpsMention]
        AND ct.SYS_CHANGE_OPERATION = ''''U''''
        AND CHANGE_TRACKING_IS_COLUMN_IN_MASK(@RpsContentColumnId, ct.SYS_CHANGE_COLUMNS) = 1
    WHERE psr.[SurveyId] = @surveyId''     
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    ---- Delete
    PRINT ''Replicating ['' + @surveySchema + ''].[Response] - Deleted''
    SET @sqlcmd = N''
    DELETE psr
    FROM ['' + @panelDatabase + ''].[dbo].[PanelistSurveyResponses] AS psr
    INNER JOIN CHANGETABLE(CHANGES '' + @surveySchema + ''.[Response], @synchronizationVersion) AS ct
        ON psr.[RespondentId] = ct.[RpsRespondent]
        AND psr.[VariableName] = ct.[RpsQuestion]
        AND psr.[Matrix] = ct.[RpsMatrix]
        AND psr.[Mention] = ct.[RpsMention]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''
    WHERE psr.[SurveyId] = @surveyId''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    ----------- Historic
    ---- Insert  
    PRINT ''Replicating ['' + @surveySchema + ''].[Historic] - Inserted'' 
    SET @sqlcmd = N''
    DECLARE @surveyActivityHistory TABLE (
        [Id] INT,
        [Index] INT,
        [DispositionCode] TINYINT
    )

    MERGE INTO ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS destination
    USING (
        SELECT
	        pr.[PanelistId],
	        h.[HisRespondent] AS [RespondentId],
	        h.[HisCallDate] AS [DateTime],
	        h.[HisCallNumber] AS [Index],
	        h.[HisResult] AS [DispositionCode]
        FROM ['' + @surveySchema + ''].[Historic] AS h
        INNER JOIN ['' + @panelDatabase +  ''].[dbo].[PanelistRespondents] pr
            ON pr.[SurveyId] = @surveyId
            AND h.[HisRespondent] = pr.[RespondentId]
        INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Historic], @synchronizationVersion) AS ct
            ON h.[HisRespondent] = ct.[HisRespondent]
            AND h.[HisCallNumber] = ct.[HisCallNumber]
            AND ct.SYS_CHANGE_OPERATION = ''''I''''
        LEFT JOIN ['' + @panelDatabase + N''].[dbo].[ActivityHistories] ah ON ah.[SurveyId] = @surveyId AND h.[HisRespondent] = ah.[RespondentId]
        LEFT JOIN  ['' + @panelDatabase + N''].[dbo].[ActivityHistories_SurveyActivityHistory] sah ON ah.[Id] = sah.[Id] AND h.[HisCallNumber] = sah.[Index]
        WHERE sah.[Id] IS NULL
    ) AS source
    ON 1 = 0 -- never match
    WHEN NOT MATCHED THEN
    INSERT ([PanelistId], [SurveyId], [RespondentId], [DateTime])
    VALUES (source.[PanelistId], @surveyId, source.[RespondentId], source.[DateTime])
    OUTPUT inserted.[Id], source.[Index], source.[DispositionCode]
    INTO @surveyActivityHistory;

    INSERT INTO ['' + @panelDatabase + ''].[dbo].[ActivityHistories_SurveyActivityHistory] ([Id], [Index], [DispositionCode])
    SELECT [Id], [Index], [DispositionCode]
    FROM @surveyActivityHistory;''   
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId     
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    -- Update
    PRINT ''Replicating ['' + @surveySchema + ''].[Historic] - Updated'' 
    SET @sqlcmd = N''
    DECLARE @HisResultColumnId int = COLUMNPROPERTY(OBJECT_ID(''''['' + @surveySchema + ''].[Historic]''''),''''HisResult'''', ''''ColumnId'''');

    UPDATE sah
    SET sah.[DispositionCode] = h.[HisResult]
    FROM ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS ah
    INNER JOIN ['' + @panelDatabase + ''].[dbo].[ActivityHistories_SurveyActivityHistory] AS sah ON ah.[Id] = sah.[Id]
    INNER JOIN ['' + @surveySchema + ''].[Historic] AS h
        ON ah.[RespondentId] = h.[HisRespondent]
        AND sah.[Index] = h.[HisCallNumber]
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Historic], @synchronizationVersion) AS ct
        ON h.[HisRespondent] = ct.[HisRespondent]
        AND h.[HisCallNumber] = ct.[HisCallNumber]
        AND ct.SYS_CHANGE_OPERATION = ''''U''''
        AND CHANGE_TRACKING_IS_COLUMN_IN_MASK(@HisResultColumnId, ct.SYS_CHANGE_COLUMNS) = 1
    WHERE ah.[SurveyId] = @surveyId''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId     
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
    -- Delete
    PRINT ''Replicating ['' + @surveySchema + ''].[Historic] - Deleted'' 
    SET @sqlcmd = N''
    DELETE ah
    FROM ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS ah
    INNER JOIN ['' + @panelDatabase + ''].[dbo].[ActivityHistories_SurveyActivityHistory] AS sah ON ah.[Id] = sah.[Id]
    INNER JOIN CHANGETABLE(CHANGES ['' + @surveySchema + ''].[Historic], @synchronizationVersion) AS ct
        ON ah.[RespondentId] = ct.[HisRespondent]
        AND sah.[Index] = ct.[HisCallNumber]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''
    WHERE ah.[SurveyId] = @surveyId''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int, @surveyId int'', @synchronizationVersion = @synchronizationVersion, @surveyId = @surveyId
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)
     
   FETCH NEXT FROM surveyCursor INTO @surveyId;
  END
  CLOSE surveyCursor
  DEALLOCATE surveyCursor

    ----------- Users   
    -- Insert
    PRINT ''Replicating [dbo].[Users] - Inserted'' 
    SET @sqlcmd = N''
    INSERT INTO ['' + @panelDatabase + ''].[dbo].[Users] ([Id], [UserName], [FirstName], [LastName], [DisplayName])
    SELECT u.[Id], u.[UserName], u.[FirstName], u.[LastName], u.[FirstName] + '''' '''' + u.[LastName]
    FROM [dbo].[Users] AS u
    INNER JOIN CHANGETABLE(CHANGES [dbo].[Users], @synchronizationVersion) AS ct
        ON u.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''I''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    -- Update  
    PRINT ''Replicating [dbo].[Users] - Updated''
    SET @sqlcmd = N''
    DECLARE @UserNameColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[Users]''''),''''UserName'''', ''''ColumnId'''');
    DECLARE @FirstNameColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[Users]''''),''''FirstName'''', ''''ColumnId'''');
    DECLARE @LastNameColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[Users]''''),''''LastName'''', ''''ColumnId'''');

    UPDATE pu
    SET pu.[UserName] = u.[UserName],
        pu.[FirstName] = u.[FirstName],
        pu.[LastName] = u.[LastName],
        pu.[DisplayName] = u.[FirstName] + '''' '''' + u.[LastName]
    FROM ['' + @panelDatabase + ''].[dbo].[Users] AS pu
    INNER JOIN [dbo].[Users] AS u ON pu.[Id] = u.[Id]
    INNER JOIN CHANGETABLE(CHANGES [dbo].[Users], @synchronizationVersion) AS ct
        ON u.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''U''''
        AND (
            CHANGE_TRACKING_IS_COLUMN_IN_MASK(@UserNameColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@FirstNameColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@LastNameColumnId, ct.SYS_CHANGE_COLUMNS) = 1
        )''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    -- Delete
    PRINT ''Replicating [dbo].[Users] - Deleted''
    SET @sqlcmd = N''
    DELETE pu
    FROM ['' + @panelDatabase + ''].[dbo].[Users] AS pu 
    INNER JOIN CHANGETABLE(CHANGES [dbo].[Users], @synchronizationVersion) AS ct
        ON pu.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    ----------- EmailInvitations
    -- Insert
    PRINT ''Replicating [dbo].[VEM_EmailInvitation] - Inserted''
    SET @sqlcmd = N''
    INSERT INTO ['' + @panelDatabase + ''].[dbo].[EmailInvitations] ([Id], [SurveyId], [Name], [TemplateId], [TemplateName], [SendDateTime], [Status])
    SELECT ei.[Id], ei.[ProjectId], ei.[Name], ei.[DefaultTemplateId], ei.[DefaultTemplateName], ei.[SendDateTime], ei.[Status]
    FROM [dbo].VEM_EmailInvitation AS ei
    INNER JOIN CHANGETABLE(CHANGES [dbo].[VEM_EmailInvitation], @synchronizationVersion) AS ct
        ON ei.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''I''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    -- Update
    PRINT ''Replicating [dbo].[VEM_EmailInvitation] - Updated''
    SET @sqlcmd = N''
    DECLARE @NameColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[VEM_EmailInvitation]''''),''''Name'''', ''''ColumnId'''');
    DECLARE @DefaultTemplateIdColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[VEM_EmailInvitation]''''),''''DefaultTemplateId'''', ''''ColumnId'''');
    DECLARE @DefaultTemplateNameColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[VEM_EmailInvitation]''''),''''DefaultTemplateName'''', ''''ColumnId'''');
    DECLARE @SendDateTimeColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[VEM_EmailInvitation]''''),''''SendDateTime'''', ''''ColumnId'''');
    DECLARE @StatusColumnId int = COLUMNPROPERTY(OBJECT_ID(''''[dbo].[VEM_EmailInvitation]''''),''''Status'''', ''''ColumnId'''');
    
    UPDATE pei
    SET pei.[Name] = ei.[Name],
        pei.[TemplateId] = ei.[DefaultTemplateId],
        pei.[TemplateName] = ei.[DefaultTemplateName],
        pei.[SendDateTime] = ei.[SendDateTime],
        pei.[Status] = ei.[Status]
    FROM ['' + @panelDatabase + ''].[dbo].[EmailInvitations] AS pei
    INNER JOIN [dbo].[VEM_EmailInvitation] AS ei ON pei.[Id] = ei.[Id]
    INNER JOIN CHANGETABLE(CHANGES [dbo].[VEM_EmailInvitation], @synchronizationVersion) AS ct
        ON ei.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''U''''
        AND (
            CHANGE_TRACKING_IS_COLUMN_IN_MASK(@NameColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@DefaultTemplateIdColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@DefaultTemplateNameColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@SendDateTimeColumnId, ct.SYS_CHANGE_COLUMNS) = 1
            OR CHANGE_TRACKING_IS_COLUMN_IN_MASK(@StatusColumnId, ct.SYS_CHANGE_COLUMNS) = 1
        )''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    -- Delete
    PRINT ''Replicating [dbo].[VEM_EmailInvitation] - Deleted''
    SET @sqlcmd = N''
    DELETE pei
    FROM ['' + @panelDatabase + ''].[dbo].[EmailInvitations] AS pei
    INNER JOIN CHANGETABLE(CHANGES [dbo].[VEM_EmailInvitation], @synchronizationVersion) AS ct
        ON pei.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    ----------- ActivityHistories_EmailActivityHistory
    -- Insert
    PRINT ''Replicating [dbo].[VEM_InvitationHistory] - Inserted''
    SET @sqlcmd = N''
    DECLARE @emailActivityHistory TABLE (
        [Id] INT,
        [EmailInvitationId] INT,
        [EmailHistoryId] INT,
        [EmailInvitationStatus] TINYINT,
        [TemplateId] INT
    )

    MERGE INTO ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS destination
    USING (
        SELECT
	        pr.[PanelistId],
	        i.[ProjectId] AS [SurveyId],
	        h.[RespondentId],
	        h.[TimeStampGMT] AS [DateTime],
	        i.[Id] AS [EmailInvitationId],
	        h.[Id] AS [EmailHistoryId],
	        CASE
		        WHEN h.[Status] = ''''Sent'''' THEN 1
		        WHEN h.[Status] = ''''Started'''' THEN 2
		        WHEN h.[Status] = ''''Bounced''''THEN 3
		        WHEN h.[Status] = ''''Completed'''' THEN 4
		        ELSE 0
	        END AS [EmailInvitationStatus],
	        i.[DefaultTemplateId] AS [TemplateId]
        FROM [dbo].[VEM_InvitationHistory] h
        INNER JOIN [dbo].[VEM_EmailInvitation] i on h.[InvitationId] = i.[Id]
        INNER JOIN ['' + @panelDatabase +  ''].[dbo].[PanelistRespondents] pr ON i.[ProjectId] = pr.[SurveyId] AND h.[RespondentId] = pr.[RespondentId]
        INNER JOIN CHANGETABLE(CHANGES [dbo].[VEM_InvitationHistory], @synchronizationVersion) AS ct
            ON h.[Id] = ct.[Id]
            AND ct.SYS_CHANGE_OPERATION = ''''I''''
    ) AS source
    ON 1 = 0 -- never match
    WHEN NOT MATCHED THEN
    INSERT ([PanelistId], [SurveyId], [RespondentId], [DateTime])
    VALUES (source.[PanelistId], source.[SurveyId], source.[RespondentId], source.[DateTime])
    OUTPUT inserted.[Id], source.[EmailInvitationId], source.[EmailHistoryId], source.[EmailInvitationStatus], source.[TemplateId]
    INTO @emailActivityHistory;

    INSERT INTO ['' + @panelDatabase + ''].[dbo].[ActivityHistories_EmailActivityHistory] ([Id], [EmailInvitationId], [EmailHistoryId], [EmailInvitationStatus], [TemplateId])
    SELECT [Id], [EmailInvitationId], [EmailHistoryId], [EmailInvitationStatus], [TemplateId]
    FROM @emailActivityHistory;''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    -- Delete  
    PRINT ''Replicating [dbo].[VEM_InvitationHistory] - Deleted''
    SET @sqlcmd = N''
    DELETE ah
    FROM ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS ah
    INNER JOIN ['' + @panelDatabase + ''].[dbo].[ActivityHistories_EmailActivityHistory] AS eah ON ah.[Id] = eah.[Id]
    INNER JOIN CHANGETABLE(CHANGES [dbo].[VEM_InvitationHistory], @synchronizationVersion) AS ct
        ON eah.[EmailHistoryId] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    ----------- ActivityHistories_SmsActivityHistory
    -- Insert
    PRINT ''Replicating [dbo].[Distributions_SMS_History] - Inserted''
    SET @sqlcmd = N''
    DECLARE @smsActivityHistory TABLE (
        [Id] INT,
        [SmsInvitationId] INT,
        [SmsInvitationStatus] TINYINT
    )

    MERGE INTO ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS destination
    USING (
        SELECT
	        pr.[PanelistId],
	        d.[SurveyId],
	        h.[RespondentId],
	        h.[TimeStamp] AS [DateTime],
	        d.[Id] AS [SmsInvitationId],
	        h.[Status] AS [SmsInvitationStatus]
        FROM [dbo].[Distributions_SMS_History] h
        INNER JOIN [dbo].[Distributions] d ON h.[DistributionId] = d.[Id]
        INNER JOIN ['' + @panelDatabase +  ''].[dbo].[PanelistRespondents] pr
            ON d.[SurveyId] = pr.[SurveyId]
            AND h.[RespondentId] = pr.[RespondentId]
        INNER JOIN CHANGETABLE(CHANGES [dbo].[Distributions_SMS_History], @synchronizationVersion) AS ct
            ON h.[Id] = ct.[Id]
            AND ct.SYS_CHANGE_OPERATION = ''''I''''
    ) AS source
    ON 1 = 0 -- never match
    WHEN NOT MATCHED THEN
    INSERT ([PanelistId], [SurveyId], [RespondentId], [DateTime])
    VALUES (source.[PanelistId], source.[SurveyId], source.[RespondentId], source.[DateTime])
    OUTPUT inserted.[Id], source.[SmsInvitationId], source.[SmsInvitationStatus]
    INTO @smsActivityHistory;

    INSERT INTO ['' + @panelDatabase + ''].[dbo].[ActivityHistories_SmsActivityHistory] ([Id], [SmsInvitationId], [SmsInvitationStatus])
    SELECT [Id], [SmsInvitationId], [SmsInvitationStatus]
    FROM @smsActivityHistory;''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    -- Delete  
    PRINT ''Replicating [dbo].[Distributions_SMS_History] - Deleted''
    SET @sqlcmd = N''
    DELETE ah
    FROM ['' + @panelDatabase + ''].[dbo].[ActivityHistories] AS ah 
    INNER JOIN ['' + @panelDatabase + ''].[dbo].[ActivityHistories_SmsActivityHistory] AS sah ON ah.[Id] = sah.[Id]
    INNER JOIN CHANGETABLE(CHANGES [dbo].[Distributions_SMS_History], @synchronizationVersion) AS ct
        ON sah.[Id] = ct.[Id]
        AND ct.SYS_CHANGE_OPERATION = ''''D''''''
    EXEC sp_executesql @sqlcmd, N''@synchronizationVersion int'', @synchronizationVersion = @synchronizationVersion
    PRINT ''Done! Affected rows: '' + CAST(@@ROWCOUNT AS NVARCHAR)

    END

    UPDATE [dbo].[ChangeTrackingVersions] SET [Updated] = GETDATE() WHERE [Version] = @synchronizationVersion
    INSERT INTO [dbo].[ChangeTrackingVersions] ([Version]) VALUES (CHANGE_TRACKING_CURRENT_VERSION())

    COMMIT TRAN
   END TRY
   BEGIN CATCH
    IF @@TRANCOUNT > 0 ROLLBACK TRAN

    DECLARE @ErrorMessage NVARCHAR(4000);
    DECLARE @ErrorSeverity INT;
    DECLARE @ErrorState INT;
    SELECT @ErrorMessage = ERROR_MESSAGE(), @ErrorSeverity = ERROR_SEVERITY(), @ErrorState = ERROR_STATE();

    UPDATE [dbo].[ChangeTrackingVersions] SET [ErrorMessage] = @ErrorMessage WHERE [Version] = @synchronizationVersion      
   
    RAISERROR (@ErrorMessage, @ErrorSeverity, @ErrorState);
   END CATCH
  END
  ELSE
  BEGIN
   PRINT ''Panel database is already up to date''
  END
END
'

declare @ClientPayload03 nvarchar(max) = N'
IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''[dbo].[A4S_sp_UpdateSurveyChangeTracking]'') AND type in (N''P'', N''PC''))
   DROP PROCEDURE [dbo].[A4S_sp_UpdateSurveyChangeTracking]
'

declare @ClientPayload04 nvarchar(max) = N'
CREATE PROCEDURE [dbo].[A4S_sp_UpdateSurveyChangeTracking]
AS
BEGIN 
  DECLARE @surveyDatabase nvarchar(64) = DB_NAME()
  DECLARE @clientId int = dbo.GetClientId(@surveyDatabase)
  DECLARE @clientIdString nvarchar(8) = CONVERT(nvarchar, @clientId)
  DECLARE @panelDatabase nvarchar(40) = ''A4Panel_Client_'' + @clientIdString
  DECLARE @sqlcmd nvarchar(max)
 
  -- Enable change tracking only for tables linked to samples  
  DECLARE surveyCursor CURSOR LOCAL FOR SELECT * FROM A4S_fn_GetPanelSurveyIds();
  DECLARE @surveyId INT;
  DECLARE @surveySchema NVARCHAR(40);
  OPEN surveyCursor;
  FETCH NEXT FROM surveyCursor INTO @surveyId;
  WHILE @@FETCH_STATUS = 0 
  BEGIN
   SET @surveySchema = ''Survey_'' + @clientIdString + ''_'' + CONVERT(NVARCHAR, @surveyId)
   IF NOT EXISTS (SELECT 1 FROM sys.change_tracking_tables WHERE object_id = OBJECT_ID(''['' + @surveySchema + ''].[Respondent]''))
	BEGIN
	   SET @sqlcmd = N''ALTER TABLE ['' + @surveySchema + ''].[Respondent] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON);
	   ALTER TABLE ['' + @surveySchema + ''].[Response] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON);
	   ALTER TABLE ['' + @surveySchema + ''].[Historic] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON);''
	END

   EXEC sp_executesql @sqlcmd
   
   FETCH NEXT FROM surveyCursor INTO @surveyId; 
  END
  CLOSE surveyCursor
  DEALLOCATE surveyCursor
END
'

declare @ClientPayload05 nvarchar(max) = N'
IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''[dbo].[A4S_sp_EnablePanelReplication]'') AND type in (N''P'', N''PC''))
   DROP PROCEDURE [dbo].[A4S_sp_EnablePanelReplication]
'

declare @ClientPayload06 nvarchar(max) = N'
EXEC dbo.sp_executesql @statement = N''CREATE PROCEDURE [dbo].[A4S_sp_EnablePanelReplication]
AS
BEGIN 
 DECLARE @surveyDatabase nvarchar(64) = DB_NAME()
 DECLARE @clientId int = dbo.GetClientId(@surveyDatabase)
 DECLARE @clientIdString nvarchar(8) = CONVERT(nvarchar, @clientId)
 DECLARE @panelDatabase nvarchar(40) = ''''A4Panel_Client_'''' + @clientIdString
 DECLARE @sqlcmd nvarchar(max)

 -- Enable change tracking for Survey database and tables
 IF NOT EXISTS(SELECT * FROM sys.change_tracking_databases WHERE database_id = DB_ID(@surveyDatabase)) 
 BEGIN
  -- Create A4S_fn_GetPanelSurveyIds
  EXEC A4S_sp_CreateGetPanelSurveyIdsFunction

  SET @sqlcmd = ''''ALTER DATABASE ['''' + @surveyDatabase + ''''] SET CHANGE_TRACKING = ON (CHANGE_RETENTION = 2 DAYS, AUTO_CLEANUP = ON);''''
  EXEC sp_executesql @sqlcmd
  
  ALTER TABLE [dbo].[Users] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
  ALTER TABLE [dbo].[VEM_EmailInvitation] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
  ALTER TABLE [dbo].[VEM_InvitationHistory] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = OFF) 
  ALTER TABLE [dbo].[Distributions_SMS_History] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = OFF) 
 END
 
 -- Create versions table
 IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N''''[dbo].[ChangeTrackingVersions]'''') AND type in (N''''U'''')) BEGIN
  CREATE TABLE [dbo].[ChangeTrackingVersions](
   [Version] [bigint] NOT NULL,
   [Updated] [datetime] NULL,
   [Reset] bit NOT NULL DEFAULT 0,
   [ErrorMessage] nvarchar(max) NULL 
  )

  INSERT INTO [dbo].[ChangeTrackingVersions] ([Version]) VALUES (CHANGE_TRACKING_CURRENT_VERSION())
 END
 ELSE BEGIN
  -- Insert current change tracking version
  DECLARE @synchronizationVersion int;
 
  SELECT TOP 1 @synchronizationVersion = [Version] FROM [ChangeTrackingVersions] ORDER BY [Version] DESC 
 
  IF @synchronizationVersion < CHANGE_TRACKING_CURRENT_VERSION()
  BEGIN
   INSERT INTO [dbo].[ChangeTrackingVersions] ([Version]) VALUES (CHANGE_TRACKING_CURRENT_VERSION())
  END
 END 

 -- Enable snapshot isolation for transactions
 SET @sqlcmd = ''''ALTER DATABASE ['''' + @surveyDatabase + ''''] SET ALLOW_SNAPSHOT_ISOLATION ON;''''
 EXEC sp_executesql @sqlcmd

 SET @sqlcmd = ''''ALTER DATABASE ['''' + @panelDatabase + ''''] SET ALLOW_SNAPSHOT_ISOLATION ON'''' 
 EXEC sp_executesql @sqlcmd
 
 -- Create job for replicate panel data
 EXEC [dbo].[A4S_sp_CreatePanelReplicationJob]
END
'' 
'

declare @ClientPayload07 nvarchar(max) = N'
IF EXISTS (SELECT 1 FROM sys.change_tracking_tables WHERE object_id = OBJECT_ID(''[dbo].[Users]'') AND is_track_columns_updated_on = 0)
BEGIN
   ALTER TABLE [dbo].[Users] DISABLE CHANGE_TRACKING
   ALTER TABLE [dbo].[Users] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
END
'

declare @ClientPayload08 nvarchar(max) = N'
IF EXISTS (SELECT 1 FROM sys.change_tracking_tables WHERE object_id = OBJECT_ID(''[dbo].[VEM_EmailInvitation]'') AND is_track_columns_updated_on = 0)
BEGIN
   ALTER TABLE [dbo].[VEM_EmailInvitation] DISABLE CHANGE_TRACKING
   ALTER TABLE [dbo].[VEM_EmailInvitation] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
END
'



declare @SurveyPayload01 nvarchar(max) = N'
--66169
IF EXISTS (SELECT 1 FROM sys.change_tracking_tables WHERE object_id = OBJECT_ID(''#SURVEY SCHEMA#.[Respondent]'') AND is_track_columns_updated_on = 0)
BEGIN
   ALTER TABLE #SURVEY SCHEMA#.[Respondent] DISABLE CHANGE_TRACKING
   ALTER TABLE #SURVEY SCHEMA#.[Respondent] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
END
'

declare @SurveyPayload02 nvarchar(max) = N'
IF EXISTS (SELECT 1 FROM sys.change_tracking_tables WHERE object_id = OBJECT_ID(''#SURVEY SCHEMA#.[Response]'') AND is_track_columns_updated_on = 0)
BEGIN
   ALTER TABLE #SURVEY SCHEMA#.[Response] DISABLE CHANGE_TRACKING
   ALTER TABLE #SURVEY SCHEMA#.[Response] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
END
'

declare @SurveyPayload03 nvarchar(max) = N'
IF EXISTS (SELECT 1 FROM sys.change_tracking_tables WHERE object_id = OBJECT_ID(''#SURVEY SCHEMA#.[Historic]'') AND is_track_columns_updated_on = 0)
BEGIN
   ALTER TABLE #SURVEY SCHEMA#.[Historic] DISABLE CHANGE_TRACKING
   ALTER TABLE #SURVEY SCHEMA#.[Historic] ENABLE CHANGE_TRACKING WITH (TRACK_COLUMNS_UPDATED = ON)
END
'

create table #DB (
	DBName nvarchar(128),
	Processed tinyint default 0
)
create table #Survey (
	DBName nvarchar(128),
	SchemaName nvarchar(128),
	Processed tinyint default 0
)
create table #SurveyVerif (
	DBName nvarchar(128),
	SchemaName nvarchar(128)
)
create table #SurveysDone (
	DBName nvarchar(128) collate database_default,
	SchemaName nvarchar(128) collate database_default
)
declare @AllDone bit = 0, 
        @TotalClients int, 
		@TotalSurveys int, 
		@ClientCounter int, 
		@SurveyCounter int, 
		@ThisClient nvarchar(128), 
		@ThisSurvey nvarchar(128), 
		@ClientCursor cursor, 
		@SurveyCursor cursor, 
		@DBSQL nvarchar(200), 
		@SQL nvarchar(max), 
		@StatusMessage varchar(max), 
		@StatusPercent tinyint

while @AllDone = 0 begin
	select @TotalClients = 0, @TotalSurveys = 0, @ClientCounter = 0, @SurveyCounter = 0
	insert into #DB 
	select 'A4Survey_Client_' + cast(Id as nvarchar(16)), 0 from Acuity4.dbo.Client
	     
    if @ProcessTemplateDB = 1 
	   insert into #DB values (N'A4Survey_Client_Template', 0)

	select @TotalClients = count(*) 
	       from #DB
	
	set @ClientCursor = cursor scroll dynamic for select DBName from #DB for update of Processed
	open @ClientCursor
	fetch next from @ClientCursor into @ThisClient
	while @@FETCH_STATUS = 0 begin
		select @ClientCounter += 1
		select @DBSQL = @ThisClient + N'.sys.sp_executesql'
		select @SQL = N'insert into #Survey select ''' + @ThisClient + ''', SchemaName, 0 from ' + @ThisClient + '.dbo.Survey except select DBName, SchemaName, 0 from #SurveysDone'
		exec @DBSQL @SQL
		select @StatusMessage = 'Updating client database "%s" (%u of %u, %u%%)', @StatusPercent = 100.0 * @ClientCounter / @TotalClients
		raiserror (@StatusMessage, 0, 1, @ThisClient, @ClientCounter, @TotalClients, @StatusPercent) with nowait

        print 'Client Payload 1'
		select @SQL = replace(replace(@ClientPayload01, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 2'
		select @SQL = replace(replace(@ClientPayload02, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 3'
		select @SQL = replace(replace(@ClientPayload03, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 4'
		select @SQL = replace(replace(@ClientPayload04, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 5'
		select @SQL = replace(replace(@ClientPayload05, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 6'
		select @SQL = replace(replace(@ClientPayload06, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 7'
		select @SQL = replace(replace(@ClientPayload07, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
        
        print 'Client Payload 8'
		select @SQL = replace(replace(@ClientPayload08, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient)
		if isnull(@SQL, '') <> '' exec @DBSQL @SQL
		print 'All Client Payload done!'

		update #DB set Processed = 1 where current of @ClientCursor
		fetch next from @ClientCursor into @ThisClient
	end

	select @TotalSurveys = count(*) from #Survey
	set @SurveyCursor = cursor forward_only for select DBName, SchemaName from #Survey for update of Processed
	open @SurveyCursor
	fetch next from @SurveyCursor into @ThisClient, @ThisSurvey
	while @@FETCH_STATUS = 0 begin
		select @SurveyCounter += 1
		select @StatusMessage = 'Updating survey "%s" (%u of %u, %u%%)', @StatusPercent = 100.0 * @SurveyCounter / @TotalSurveys
		raiserror (@StatusMessage, 0, 1, @ThisSurvey, @SurveyCounter, @TotalSurveys, @StatusPercent) with nowait

		declare @SurveyExists bit = 0;
		select @DBSQL = @ThisClient + N'.sys.sp_executesql'

		select @SQL = N'SELECT @SurveyExistsOutput = 1; if exists (select * from ' + @ThisClient + '.sys.schemas where [name] = ''' + @ThisSurvey + ''') SELECT @SurveyExistsOutput = 1 '
		exec sp_executesql @SQL, N'@SurveyExistsOutput bit out', @SurveyExists out
		--select @SurveyExists

		if @SurveyExists = 1
		BEGIN 
		   print 'Survey Payload 1'
		   SELECT @SQL = replace(replace(replace(@SurveyPayload01, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient), '#SURVEY SCHEMA#', @ThisSurvey)
		   IF isnull(@SQL, '') <> '' exec @DBSQL @SQL

		   print 'Survey Payload 2'
		   SELECT @SQL = replace(replace(replace(@SurveyPayload02, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient), '#SURVEY SCHEMA#', @ThisSurvey)
		   IF isnull(@SQL, '') <> '' exec @DBSQL @SQL

		   print 'Survey Payload 3'
		   SELECT @SQL = replace(replace(replace(@SurveyPayload03, '#CLIENT ID#', substring(@ThisClient, 17, 8)), '#CLIENT DB#', @ThisClient), '#SURVEY SCHEMA#', @ThisSurvey)
		   IF isnull(@SQL, '') <> '' exec @DBSQL @SQL
		   print 'All Survey Payload done!'

        END
        ELSE 
		BEGIN
		   print '  >>>>> Survey has been deleted!'
		END

		update #Survey set Processed = 1 where current of @SurveyCursor
		fetch next from @SurveyCursor into @ThisClient, @ThisSurvey
	end

	truncate table #SurveyVerif
	fetch first from @ClientCursor into @ThisClient
	while @@FETCH_STATUS = 0 begin
		select @DBSQL = @ThisClient + N'.sys.sp_executesql'
		select @SQL = N'insert into #SurveyVerif select ''' + @ThisClient + ''', SchemaName from ' + @ThisClient + '.dbo.Survey except select DBName, SchemaName from #SurveysDone'
		exec @DBSQL @SQL
		fetch next from @ClientCursor into @ThisClient
	end

	close @ClientCursor
	close @SurveyCursor
	deallocate @ClientCursor
	deallocate @SurveyCursor

-- Backup client template db if it was updated
	IF @ProcessTemplateDB = 1 
    BEGIN
      PRINT 'Creating backup for A4Survey_Client_Template';
      SET @SQL = 'USE Acuity4;
                  exec [dbo].[VX_sp_BackupClientTemplateDatabase]'
      EXEC (@SQL); 
    END

	IF exists (select DBName, SchemaName from #SurveyVerif except select DBName, SchemaName from #Survey where Processed = 1) 
	BEGIN
		raiserror ('  >>>>> New survey(s) have been added or were skipped, going again!', 0, 1)
		insert into #SurveysDone select DBName, SchemaName from #Survey where Processed = 1
		truncate table #DB
		truncate table #Survey
	END 
	ELSE 
    BEGIN
	  SELECT @AllDone = 1
    END

end



--select * from #DB
--select * from #Survey

drop table #DB
drop table #Survey
drop table #SurveyVerif
drop table #SurveysDone